/* PROJECT-LEVEL WOMEN'S EMPOWERMENT IN AGRICULTURE INDEX (PRO-WEAI) v1.1
	TABLES AND CHARTS DO-FILE
	UDPATED AUGUST 2018
	CONTACT: IFPRI-WEAI@CGIAR.ORG
	
	THIS DO-FILE CREATES PRO-WEAI TABLES AND CHARTS. THESE ARE SUGGESTED (NOT REQUIRED) WAYS TO DESCRIBE PRO-WEAI RESULTS. 
	NOTE: THIS DO-FILE IS BASED ON A DRAFT VERSION OF THE PRO-WEAI AND IS SUBJECT TO CHANGE. 
	
	UPDATES:
	(v1.1) April 2020: Correct dataset for indicator bar charts: line 38		
		   May 2021: Changed dataset for indicator bar charts to calculate file: line 38 
		   July 2021: Edited table to include empoweremt score 
		   May 2022: Edited to mute optional indicators Respect Among HH Members and Membership in Influential Groups*/

	
********************************************************************************


clear all
set more off

********************************************************************************
					*** SET DIRECTORY ***
********************************************************************************

	** Set working directory
	cd "your/working/directory"

	** Store project name
	global projectname "your_project_name"

********************************************************************************
					*** INDICATOR BAR CHARTS *** 
********************************************************************************

/* This section creates bar charts (in PNG format) comparing the percentage of
	respondents adequate in each pro-WEAI indicator by gender and household 
	type.		*/

	use "$projectname_proweai_calculate", replace

	gen hh_type_sex=.
		replace hh_type_sex=1 if (hh_type==1 & sex==1)
		replace hh_type_sex=2 if (hh_type==1 & sex==2)
		replace hh_type_sex=3 if (hh_type==2 & sex==2)
	lab def hh_type_sex_val 1"Dual male" 2"Dual female" 3"Female only"
	lab val hh_type_sex hh_type_sex_val
	lab var hh_type_sex "Gender of respondent by household type"

	bys hh_type_sex: gen freq=_N
	local var autonomy_inc selfeff never_violence feelinputdecagr assetownership ///
		credit_accdec incomecontrol work_balance mobility groupmember /*group_inf respect*/ 
	collapse (sum) `var', by(hh_type_sex freq)
	collapse (sum) `var' freq, by(hh_type_sex)
	foreach x in `var' {
		gen `x'_pct = (`x'/freq)*100
		format %3.2f `x'_pct
		}
		
	foreach x in ///
		autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec ///
		incomecontrol work_balance mobility groupmember /*group_inf respect*/ {
		
		graph hbar (sum) `x'_pct, over(hh_type_sex, label(labsize(vsmall)) ///
		relabel(1 "Men" 2 "Women, dual-adult" 3"Women, female-only")) ///
		asyvars ytitle(Percent adequate) bargap(20) ///
		ytitle(, size(medium) margin(medsmall)) yscale(lcolor(black) line) ///
		ylabel(0(20)100, labsize(small) grid glcolor(gs8) gmin gmax gextend) yscale(alt) ///
		legend(rows(1) size(small) nobox region(fcolor(white) lcolor(white))) scheme(sj) ///
		graphregion(fcolor(white) lcolor(none)) bar(1,color(emerald)) bar(2,color(purple)) bar(3,color(dkorange))
		graph export "indicator_`x'.png", replace
		}

********************************************************************************
					*** PRO-WEAI TABLE *** 
********************************************************************************

/* This section creates a table (in Excel) of pro-WEAI and its component 
	statistics.		*/

	use "$projectname_proweai_calculate", clear

	mean No_obs PROWEAI_3DE emp_score achieve_emp mean3DE_unemp GPI dahh achieve_GP hh_ineq I_GPI PROWEAI if sex==2
	scalar m = round(el(r(table),1,1),.01)			
	mat A=round(el(r(table),1,1),.01)\round(el(r(table),1,2),.01)\round(el(r(table),1,3),.01)\round(el(r(table),1,4),.01)\round(el(r(table),1,5),.01)\round(el(r(table),1,6),.01)\round(el(r(table),1,7),.01)\round(el(r(table),1,8),.01)\round(el(r(table),1,9),.01)\round(el(r(table),1,10),.01)\round(el(r(table),1,11),.01)
	mat list A

	mean No_obs PROWEAI_3DE emp_score achieve_emp mean3DE_unemp if sex==1
	scalar m = round(el(r(table),1,1),.01)			
	mat B=round(el(r(table),1,1),.01)\round(el(r(table),1,2),.01)\round(el(r(table),1,3),.01)\round(el(r(table),1,4),.01)\round(el(r(table),1,5),.01)
	mat list B

	putexcel set "proweai_charts_$projectname.xlsx", sheet("Pro-WEAI Table") modify
	putexcel A1:C1="Pro-WEAI", merge font(calibri,11,black) bold border(bottom,double,black)
	putexcel B3=matrix(A) C3=matrix(B), vcenter hcenter
	putexcel A2="Indicator" B2="Women" C2="Men", border(bottom,thin,black) font(calibri,11,black) vcenter hcenter
	putexcel A4="3DE score" A8="Gender Parity Index (GPI)" A13="Pro-WEAI", ///
		bold font(calibri,11,black) 
	putexcel A3="Number of observations" A5="Empowerment score"  A6="% achieving empowerment" ///
		A7="Mean 3DE score for not yet empowered" A9="Number of dual-adult households" ///
		A10="% achieving gender parity" A11="Average intra-household inequality score" ///
		A12="Empowerment gap", font(calibri,11,black)
	putexcel A14:C14="", merge font(calibri,11,black) border(top,double,black)

	clear

********************************************************************************
					*** PRO-WEAI GRAPH *** 
********************************************************************************

/* This section creates an automated Stata version of the pro-WEAI graph and
		exports the graph in PNG format */
	
	use "$projectname_proweai_calculate", clear

	* Keep only main variables needed for graph
	keep sex disempowered autonomy_inc_cont_25_DAI selfeff_cont_25_DAI never_violence_cont_25_DAI /*respect_cont_25_DAI*/ feelinputdecagr_cont_25_DAI assetownership_cont_25_DAI credit_accdec_cont_25_DAI incomecontrol_cont_25_DAI work_balance_cont_25_DAI mobility_cont_25_DAI groupmember_cont_25_DAI /*group_inf_cont_25_DAI*/
	
	gl pweai1 "autonomy_inc selfeff never_violence  feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember" /*add respect and/or group_inf if used*/

	* Rename to shorter variable names
	foreach var in $pweai1 {
		
		ren `var'_cont_25_DAI `var'
	}
	
	* Collapse (means)
	collapse $pweai1 disempowered, by(sex)

	* Generate contribution to total disempowerment
	foreach var in $pweai1 {

		bys sex: gen c_`var'=(`var'*disempowered)
		la var c_`var' "Contribution to disempowerment"
	}

	* Prep for graph

	gl pweai2 "c_autonomy_inc c_selfeff c_never_violence c_feelinputdecagr c_assetownership c_credit_accdec c_incomecontrol c_work_balance c_mobility c_groupmember" /*add c_respect and/or c_group_inf if used */
	recode sex (2=0)
	la define sex_lab 0"Women" 1"Men"
	la val sex sex_lab
	
	* Graph (stacked bar graph)
		*remove /* */ around bar 4 and lab 4 if using Respect Among HH Members*
		*remove /* */ around bar 12 and lab 12 if using Membership in Influential Groups*
	
	gr bar $pweai2, over(sex) stack ///
		yreverse xalternate graphregion(color(white)) ///
		bar(1, fc("186 228 179") lc(white) lw(vvthin)) bar(2, fc("116 196 118") lc(none) lw(vvthin)) bar(3, fc("49 163 84") lc(none) lw(vvthin)) ///
		/*bar(4, fc("0 109 44") lc(none) lw(vvthin))*/ bar(5, fc("254 237 222") lc(none) lw(vvthin)) bar(6, fc("253 208 162") lc(none) lw(vvthin)) ///
		bar(7, fc("253 174 107") lc(none) lw(vvthin)) bar(8, fc("253 141 60") lc(none) lw(vvthin)) bar(9, fc("230 85 13") lc(none) lw(vvthin)) ///
		bar(10, fc("166 54 3") lc(none) lw(vvthin)) bar(11, fc("188 189 220") lc(none) lw(vvthin)) /*bar(12, fc("117 107 177") lc(none) lw(vvthin))*/ ///
		legend(cols(1) pos(3) si(vsmall) lw(vvthin) region(lwidth(none)) ///
		lab(1 "Autonomy in decision making") lab(2 "Self-efficacy") lab(3 "Attitudes towards domestic violence") ///
		/*lab(4 "Respect among household members")*/ lab(5 "Input in productive decisions") lab(6 "Ownership of land and other assets") ///
		lab(7 "Access to and decisions on credit") lab(8 "Control over use of income") lab(9 "Work balance") ///
		lab(10 "Visiting important locations") lab(11 "Group membership") /*lab(12 "Membership in influential groups"))*/ ///
		title("Contributions to disempowerment", span si(medium)) ///
		ytitle("Total disempowerment", si(medium)) ///
		yla(, format(%9.1f)) yscale(titlegap(*10))
		
	gr export "Pro-WEAI bar graph.png", as(png) replace

	clear

********************************************************************************
